<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\SystemSetting;

class ManageSettings extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'settings:manage 
                            {action : Ação a ser executada (get, set, list, remove)}
                            {key? : Chave da configuração}
                            {value? : Valor da configuração}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Gerencia as configurações do sistema';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $action = $this->argument('action');
        $key = $this->argument('key');
        $value = $this->argument('value');

        switch ($action) {
            case 'get':
                $this->getSetting($key);
                break;
            case 'set':
                $this->setSetting($key, $value);
                break;
            case 'list':
                $this->listSettings();
                break;
            case 'remove':
                $this->removeSetting($key);
                break;
            default:
                $this->error('Ação inválida. Use: get, set, list ou remove');
                return 1;
        }

        return 0;
    }

    private function getSetting($key)
    {
        if (!$key) {
            $this->error('Chave é obrigatória para a ação get');
            return;
        }

        $value = SystemSetting::getValue($key);
        
        if ($value === null) {
            $this->warn("Configuração '{$key}' não encontrada");
        } else {
            $this->info("{$key}: {$value}");
        }
    }

    private function setSetting($key, $value)
    {
        if (!$key || !$value) {
            $this->error('Chave e valor são obrigatórios para a ação set');
            return;
        }

        $description = $this->ask('Descrição da configuração (opcional):');
        $type = $this->choice('Tipo da configuração:', ['text', 'number', 'boolean', 'url', 'email'], 'text');

        $success = SystemSetting::setValue($key, $value, $description, $type);
        
        if ($success) {
            $this->info("Configuração '{$key}' definida com sucesso!");
        } else {
            $this->error('Erro ao definir configuração');
        }
    }

    private function listSettings()
    {
        $settings = SystemSetting::where('is_active', true)->get();

        if ($settings->isEmpty()) {
            $this->warn('Nenhuma configuração encontrada');
            return;
        }

        $this->info('Configurações do sistema:');
        $this->newLine();

        $headers = ['Chave', 'Valor', 'Descrição', 'Tipo'];
        $rows = [];

        foreach ($settings as $setting) {
            $rows[] = [
                $setting->key,
                $setting->value,
                $setting->description ?? '-',
                $setting->type
            ];
        }

        $this->table($headers, $rows);
    }

    private function removeSetting($key)
    {
        if (!$key) {
            $this->error('Chave é obrigatória para a ação remove');
            return;
        }

        if ($this->confirm("Tem certeza que deseja remover a configuração '{$key}'?")) {
            $success = SystemSetting::remove($key);
            
            if ($success) {
                $this->info("Configuração '{$key}' removida com sucesso!");
            } else {
                $this->warn("Configuração '{$key}' não encontrada");
            }
        }
    }
}
