<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class SystemSetting extends Model
{
    use HasFactory;

    protected $fillable = [
        'key',
        'value',
        'description',
        'type',
        'is_active'
    ];

    protected $casts = [
        'is_active' => 'boolean',
    ];

    /**
     * Busca uma configuração pelo nome da chave
     */
    public static function getValue(string $key, $default = null)
    {
        // Tenta buscar do cache primeiro
        $cacheKey = "system_setting_{$key}";
        
        return Cache::remember($cacheKey, 3600, function () use ($key, $default) {
            $setting = self::where('key', $key)
                          ->where('is_active', true)
                          ->first();
            
            return $setting ? $setting->value : $default;
        });
    }

    /**
     * Define o valor de uma configuração
     */
    public static function setValue(string $key, $value, string $description = null, string $type = 'text'): bool
    {
        $setting = self::updateOrCreate(
            ['key' => $key],
            [
                'value' => $value,
                'description' => $description,
                'type' => $type,
                'is_active' => true
            ]
        );

        // Limpa o cache
        Cache::forget("system_setting_{$key}");
        
        return $setting->exists;
    }

    /**
     * Remove uma configuração
     */
    public static function remove(string $key): bool
    {
        $deleted = self::where('key', $key)->delete();
        
        if ($deleted) {
            Cache::forget("system_setting_{$key}");
        }
        
        return $deleted > 0;
    }

    /**
     * Limpa todo o cache de configurações
     */
    public static function clearCache(): void
    {
        $settings = self::all();
        
        foreach ($settings as $setting) {
            Cache::forget("system_setting_{$setting->key}");
        }
    }
}
