#!/bin/bash

# Script de Deploy para Laravel
# Uso: ./deploy.sh

echo "🚀 Iniciando deploy do Laravel..."

# Cores para output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Função para imprimir mensagens coloridas
print_message() {
    echo -e "${GREEN}$1${NC}"
}

print_warning() {
    echo -e "${YELLOW}$1${NC}"
}

print_error() {
    echo -e "${RED}$1${NC}"
}

# Verificar se estamos no diretório correto
if [ ! -f "artisan" ]; then
    print_error "❌ Erro: Execute este script na raiz do projeto Laravel"
    exit 1
fi

print_message "📋 Verificando ambiente..."

# Verificar se o Composer está instalado
if ! command -v composer &> /dev/null; then
    print_error "❌ Composer não encontrado. Instale o Composer primeiro."
    exit 1
fi

# Verificar se o PHP está instalado
if ! command -v php &> /dev/null; then
    print_error "❌ PHP não encontrado. Instale o PHP primeiro."
    exit 1
fi

print_message "🧹 Limpando caches..."

# Limpar caches
php artisan config:clear
php artisan cache:clear
php artisan view:clear
php artisan route:clear

print_message "⚙️ Otimizando para produção..."

# Otimizar para produção
php artisan config:cache
php artisan route:cache
php artisan view:cache

print_message "📦 Instalando dependências..."

# Instalar dependências do Composer (sem dev)
composer install --optimize-autoloader --no-dev

# Verificar se existe package.json
if [ -f "package.json" ]; then
    print_message "📦 Instalando dependências do Node.js..."
    
    # Verificar se o npm está instalado
    if command -v npm &> /dev/null; then
        npm install --production
        npm run build
    else
        print_warning "⚠️ NPM não encontrado. Pule esta etapa se não usar assets compilados."
    fi
fi

print_message "🔐 Verificando chave da aplicação..."

# Verificar se a chave da aplicação existe
if [ -z "$(grep 'APP_KEY=base64:' .env 2>/dev/null)" ]; then
    print_warning "⚠️ Chave da aplicação não encontrada. Gerando nova chave..."
    php artisan key:generate
fi

print_message "🗄️ Executando migrações..."

# Executar migrações
php artisan migrate --force

print_message "📊 Executando seeders (se existirem)..."

# Executar seeders se existirem
if [ -d "database/seeders" ] && [ "$(ls -A database/seeders)" ]; then
    php artisan db:seed --force
fi

print_message "📁 Configurando permissões..."

# Configurar permissões
chmod -R 755 storage/
chmod -R 755 bootstrap/cache/
chmod 755 public/

print_message "✅ Deploy concluído com sucesso!"

echo ""
print_message "📋 Próximos passos:"
echo "1. Configure o arquivo .env no servidor"
echo "2. Configure o Document Root para a pasta public/"
echo "3. Configure o .htaccess se necessário"
echo "4. Teste a aplicação"

echo ""
print_warning "⚠️ Lembre-se de NÃO enviar para o servidor:"
echo "- /vendor/"
echo "- /node_modules/"
echo "- /storage/ (conteúdo)"
echo "- /bootstrap/cache/ (conteúdo)"
echo "- .env (criar novo no servidor)"
echo "- .git/"

echo ""
print_message "📚 Consulte o arquivo docs/DEPLOY_GUIDE.md para instruções detalhadas" 