# Sistema de Configurações

Este sistema permite gerenciar configurações gerais do sistema através do banco de dados, facilitando a manutenção e atualização de parâmetros sem necessidade de alterar código.

## Estrutura

### Tabela `system_settings`
- `id` - ID único
- `key` - Chave única da configuração
- `value` - Valor da configuração
- `description` - Descrição da configuração
- `type` - Tipo do valor (text, number, boolean, url, email)
- `is_active` - Se a configuração está ativa
- `created_at` e `updated_at` - Timestamps

## Como Usar

### 1. Função Helper `setting()`

```php
// Buscar uma configuração
$jiraUrl = setting('jira_url', 'https://default-url.com');

// Em views Blade
<a href="{{ setting('jira_url') }}">Link do Jira</a>
```

### 2. Função Helper `set_setting()`

```php
// Definir uma configuração
set_setting('jira_url', 'https://novo-jira.com', 'URL do sistema Jira', 'url');
```

### 3. Modelo SystemSetting

```php
use App\Models\SystemSetting;

// Buscar valor
$value = SystemSetting::getValue('jira_url');

// Definir valor
SystemSetting::setValue('jira_url', 'https://novo-jira.com');

// Remover configuração
SystemSetting::remove('jira_url');
```

## Comando Artisan

### Listar todas as configurações
```bash
php artisan settings:manage list
```

### Buscar uma configuração
```bash
php artisan settings:manage get jira_url
```

### Definir uma configuração
```bash
php artisan settings:manage set jira_url "https://novo-jira.com"
```

### Remover uma configuração
```bash
php artisan settings:manage remove jira_url
```

## Configurações Padrão

O seeder `SystemSettingsSeeder` cria as seguintes configurações iniciais:

- `jira_url` - URL do sistema Jira
- `support_email` - E-mail de contato do suporte
- `company_name` - Nome da empresa
- `system_version` - Versão do sistema

## Cache

O sistema utiliza cache para melhorar a performance. As configurações são armazenadas em cache por 1 hora. Para limpar o cache:

```php
SystemSetting::clearCache();
```

## Exemplos de Uso

### No Login (já implementado)
```php
<a href="{{ setting('jira_url', 'https://jira.exemplo.com') }}" target="_blank">Jira</a>
```

### Em Controllers
```php
public function index()
{
    $companyName = setting('company_name', 'Empresa Padrão');
    $supportEmail = setting('support_email');
    
    return view('dashboard', compact('companyName', 'supportEmail'));
}
```

### Em Configurações
```php
// config/app.php
'name' => setting('company_name', env('APP_NAME', 'Laravel')),
```

## Tipos de Configuração

- `text` - Texto simples
- `number` - Números
- `boolean` - Verdadeiro/Falso
- `url` - URLs
- `email` - Endereços de e-mail

## Boas Práticas

1. Sempre forneça um valor padrão ao usar `setting()`
2. Use chaves descritivas e consistentes
3. Documente as configurações com descrições claras
4. Use o tipo correto para cada configuração
5. Mantenha as configurações organizadas por categoria 